precision highp float;

uniform vec2 X[8];

mat3 bike1tm;
mat3 bike2tm;
mat3 camtm;

vec3 bike1pos;
vec3 bike2pos;
vec3 campos;
vec3 rO;
vec3 rD;
vec3 hitNormal;

float time;
float timeScaled;
float pulse;
float hDistTemp;

int hMaterial;
int hMaterialTemp;

// ##################### Math ##################### 

// Mod around 0, might have been smarter to move entire scene further from 0,0,0 to make it simpler
void R(inout float x, float n) {
	x = mod(abs(x),2.*n);
	x = x>n ? 2.*n-x : x;
}

vec3 M(vec3 v) {
	v.z+=1.;
	
	R(v.x,8.);
	R(v.y,8.);
	R(v.z,4.);
	
	if (v.x < v.y) {
		float x = v.x;
		v.x = v.y;
		v.y = x;
	}
	
	return v;
}

// ##################### Booleans ##################### 

void U(inout float ad, inout int am, float bd, int bm) {
	if (abs(ad)>abs(bd))
		am = bm;
	ad = max(ad,-bd);
}

void E(float ad, int am, float bd, int bm) {
	float distance = max(ad,bd);
	if (hDistTemp > distance) {
		hDistTemp = distance;
		hMaterialTemp = abs(ad)<abs(bd)?am:bm;
	}
}

// ##################### Primitives ##################### 

float B(vec3 p, vec3 b) {
	vec3 v = abs(p) - b;
	return min(v.x>v.y?v.x>v.z?v.x:v.z:v.y>v.z?v.y:v.z,length(max(v,.0)));
}

float O(vec3 p, vec2 t) {
	return length(vec2(length(p.xz)-t.x,p.y))-t.y;
}

// ##################### Scene ##################### 

void K(mat3 tm, vec3 pos, vec3 p) { // Bike distance field
	p -= pos;
	float distance = length(p);
	if (distance>.4) {
		distance -= .1;
		if (hDistTemp > distance)
			hDistTemp = distance;
	} else {
		vec3 vp = p * tm;
		vp = vec3(abs(vp.xy), vp.z+.04);
		
		distance = O(vp-vec3(0,0,-.07), vec2(.15, .05));
		int material = 4;
		U(distance, material, length(vp - vec3(.23,0, .022))-.16, 4);
		U(distance, material, length(vp - vec3(0,0,-.15))- .2, 3);
		U(distance, material, O(vp - vec3(.228,0,0), vec2(.16, .038)), 3);
		U(distance, material, O(vp - vec3(0,.043,-.07), vec2(.19, .012)), 2);
		if (hDistTemp > distance) {
			hDistTemp = distance;
			hMaterialTemp = material;
		}
		vp-=vec3(.1,0,.042);
		E(O(vp - vec3(0,.031,0), vec2(.038, .011)), 3, O(vp, vec2(.03)), 4);
		E(O(vp, vec2(.15, .1)), 2, length(vp) - .06, 2);
	}
}

float F(vec3 p) { // Distance field
	vec3 mp = M(p),
	cp = mod(mp, vec3(1)) - .5;
	
	// Buildings
	float width = .39 + sin(time*2.) * .1;
	hDistTemp = max( B(mp-vec3(8,8,0),vec3(4,4,8)), 
					mix(length(cp) - width,
						B(cp, vec3(width)),
						(sin(time) + 1.) * .5));
	hMaterialTemp = 1;

	if (timeScaled > 35. && timeScaled < 142.) { // Bikes
		K(bike1tm, bike1pos, p);
		K(bike2tm, bike2pos, p);
	}

	if (timeScaled > 25. && timeScaled < 150.) // Road
		E(B(mp-vec3(0,0,1),vec3(8,1,.005)), 0,
			length(vec3(mod(mp.x, .1) - .05,mod(mp.y, .1) - .05,mp.z - .98)) - .045, 0);

	return hDistTemp;
}

// ##################### Raymarching ##################### 

float Y() {
	float minStep = .0005, dist = .05;
	for (int i = 0; i < 156; i++) {
		vec3 p = rO + rD * dist;
		float f = F(p);
		if (f <  .0002) {
			hMaterial = hMaterialTemp;
			hitNormal = normalize(vec3(
				f - F(p - vec3(.001, 0, 0)),
				f - F(p - vec3(0, .001, 0)),
				f - F(p - vec3(0, 0, .001))));
			return dist;
		}
		dist += f < minStep ? minStep : f;
		if (dist > 40.)
			break;
		minStep += 0.0000018 * float(i*i);
	}
	return 40.;
}

// ##################### Shading ##################### 

float P(vec3 light) {
	light.z += .07;
	vec3 lightDir = light - rO;
	float distance = length( lightDir );
	distance *= distance;
	return pow(clamp(dot(hitNormal, normalize(lightDir/distance + rD)),0.,1.), 20. ) / distance;
}

float D(vec3 light, float sharpness) {
	return pow(clamp(dot(hitNormal, normalize(light-rO)), 0., 1.), sharpness);
}

vec3 S() { // Shading happens here
	vec3 mp = M(rO), color = vec3(1.2, .4, .25), floored = floor(rO);

	if (hMaterial == 1) { // Buildings
		vec2 fp = floor(rO.xy/(16.));
		
		float r = fract(sin(dot(fp,vec2(12.9898,78.233))) * 43758.5453);
		float r2 = fract(sin(dot(vec2(floored.x, floored.y + floored.z * 13.),
					vec2(12.9898,78.233))) * 43758.5453);

		color = mix(vec3(1.5,0.,1.5), vec3(.8,.8,1.2), r);
		
		if ((mp.x < 5. && mod(rO.z+floor(r*6.9),6.) > 2.)) // Corners
			color *= 2.;
		else if (r2 > .98) // Bright spots
			color = vec3(2);
		else // Standard
			color *= (pulse + .7 * pow(r2, 4.)) * (2.0 - floor(mp.x) / 4.);
		
		color += (D(bike1pos, 40.) * vec3(1.2, .4, .25) +
				 D(bike2pos, 40.) * vec3(.9,.9,.1) +
				 D(campos, 40.) * vec3(0,1,1)) * .2;
		
		return color;
	}
		
	if (length(rO-bike1pos) > .5) // Change color around bike 2
		color = vec3(.95,.85,.1);

	if (hMaterial == 2) // Tires
		return color * 2.;
	
	if (hMaterial > 2) // Bike innards + Reflection second time + Inner tire
		return color * (dot(hitNormal, rD) + 1.) * .5;
	
	color = vec3(0);

	if (timeScaled > 35. && timeScaled < 142.)
		color = ((P(bike1pos - bike1tm[0]*.1) +
				P(bike1pos + bike1tm[0]*.1)) * vec3(1.2, .4, .25) +
				(P(bike2pos - bike2tm[0]*.1) +
				P(bike2pos + bike2tm[0]*.1)) * vec3(.9,.9,.1))
				* .08;
	
	if (mp.y > .9 || (mp.y > .8 && (mod(mp.x/.5,2.) < 1.))) // Outline + Stripes
		color += vec3(0,2,2);
	if ((mp.y > .4 && mp.y < .5)) // Line
		color += vec3(1);
	
	color += D(campos, 1.) * vec3(.5,0,.5) * pulse;
	
	return color;
}

// ##################### Cameras ##################### 

vec3 C(float fov) { // Fisheye
	vec2 pos = (gl_FragCoord.xy-vec2(1280,720)*.5)/720.; // Screen space: -.5 .. .5
	float radius = length(pos);
	float theta = radius * 3.14159 * fov;
	return vec3(cos(theta),(pos.x/radius)*sin(theta),(pos.y/radius)*sin(theta));
}

mat3 I(vec3 dir, float tilt) { // Transformation matrix
	vec3 vx = normalize(dir);
	vec3 vy = normalize(vec3(-vx.y, vx.x, tilt));
	return mat3(vx, vy, cross(vx, vy));
}

float A(float timePos) { // Flash screen
	return max(0., 4. - abs(timeScaled-timePos) * 20.);
}

// ##################### Main ##################### 

void main( void ) {
	time = X[7].y;
	timeScaled = time * 2.5;

	// Pulse is synced with audio
	pulse = mod(2.4 * (timeScaled+.12),1.) * 5.;
	if (pulse>1.)
		pulse = 1.25 - pulse * .25;
	pulse *= pulse;
	pulse = pulse * clamp(6.9-abs(80.-timeScaled)*.1,0.,1.);
	
	// Calculate transformation matrices for bikes and camera
	camtm   = I(vec3(X[5],0), X[7].x);
	bike1tm = I(vec3(X[1],0), X[6].x-cos(time)*.35);
	bike2tm = I(vec3(X[3],0), X[6].y-sin(time)*.35);
	
	// Positions for bikes and camera
	campos   = vec3(X[4]*16. + camtm[1].xy   * sin(time)  * 1.5, 1.2+sin(time*.5+2.5));
	bike1pos = vec3(X[0]*16. + bike1tm[1].xy * sin(time) * .3,   .07);
	bike2pos = vec3(X[2]*16. + bike2tm[1].xy * cos(time) * .3,   .07);
	
	rO = campos;

	vec3 color = vec3(0);
	
	// This had to be split into two parts, because otherwise it caused out of memory error when
	// compiling, I guess there might be some neater way of doing this
	int camera = // 0=free, 1=bike, 2=drive by, 3=follow, 4=warp, 5=sky, 6=rear view
		timeScaled < 45. ? 0 :
		timeScaled < 52. ? 1 :
		timeScaled < 55. ? 3 :
		timeScaled < 59. ? 5 :
		timeScaled < 63.6 ? 6 :
		timeScaled < 67.5 ? 1 :
		timeScaled < 70.5 ? 2 :
		timeScaled < 73.5 ? 3 :
		timeScaled < 76. ? 0 :
		timeScaled < 87. ? 1 : 2;
	if (timeScaled > 93.)
		camera =
		timeScaled < 98. ? 0 :
		timeScaled < 101. ? 5 :
		timeScaled < 106. ? 3 :
		timeScaled < 112. ? 4 :
		timeScaled < 115. ? 2 :
		timeScaled < 120. ? 3 :
		timeScaled < 125. ? 1 :
		timeScaled < 130. ? 3 :
		timeScaled < 133. ? 4 :
		timeScaled < 135. ? 5 :
		timeScaled < 137. ? 6 :
		timeScaled < 138. ? 3 :
		timeScaled < 145. ? 1 : 0;
	
	if (camera==2) { // Drive by cam
		rO = timeScaled < 80. ? 
				vec3(-112, -81.6, .4) : // 1st
				timeScaled < 100. ? 
					vec3(-192, -113.6, 1.2) : // 2nd
					vec3(-81.6, -78.4, -.01); // 3rd
		rD = I(bike1pos-rO,0.) * C(.15);
	} 
	if (camera==1) { // Bike cam
		float radius = .25 + sin(timeScaled) * .15;
		rO = vec3(radius * cos(time*1.5), radius * sin(time*1.5), .1) + bike1pos;
		rD = I(bike1pos-rO,0.) * C(.4);
	} 
	if (camera==3) { // Follow cam
		rO = bike2pos + vec3(0,0,.05) + vec3(normalize(bike2tm[1].xy)*.13, 0);
		rD = I(bike1pos-rO,0.) * C(.6);
	} 
	if (camera==6) { // Rear view cam
		rO = bike1pos + vec3(0,0,.01) + vec3(normalize(bike2tm[1].xy)*.05, 0) + bike1tm[0]*.25;
		rD = I(-bike1tm[0],0.) * C(.6);
	}
	if (camera==4) // Warp cam
		rD = I(bike1pos-rO,0.) * C(.8);
		
	if (camera==5) { // Sky cam
		rO = (bike1pos + bike2pos) * .5 + vec3(.2,.2,2);
		rD = -C(.8).yzx;
	}
	if (camera==0) // Free cam
		rD = camtm * C(max(timeScaled*.1-14.9,.5));
	
	float dist = Y(); // Raymarch into scene
	if (dist != 40.) {
		rO += rD * dist;
		if (hMaterial == 4) {
			rD = reflect(rD, hitNormal);
			float dist2 = Y(); // Raymarch reflection
			if (dist2 != 40.) {
				rO += rD * dist2;
				color = S();
			}
			
			color = mix(color, vec3(0,.2,.2) * (rD.z+1.) * (rD.z+1.), min(dist2 / 40., 1.))
						* vec3(.8,.5,.8); // Sky
		} else
			color = S();
	}
	
	color = mix(color, vec3(0,.2,.2) * (rD.z+1.) * (rD.z+1.), min(dist / 40., 1.)); // Sky
	
	if (.25-min(abs(1.-timeScaled),abs(127.-timeScaled))*.04+sin(timeScaled*400.)*.02 > .0)
		color = vec3(pow((color.x+color.y+color.z)*.33,1.)); // Black and white

	// Add screen flashes to final color
	gl_FragColor = vec4((timeScaled > 159.5 ?
							vec3(0) :
							timeScaled < 2. ?
							vec3(0) :
							color
							+ vec3(1,.1,1) * A(25.)
							+ vec3(.1,1,1) * A(35.)
							+ vec3(.1,1,1) * A(142.)
							+ vec3(1,.1,1) * A(150.))
						+ vec3(1,1,.1) * A(159.5)
						+ vec3(1,1,.1) * A(2.)
						,1);
}
