/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#include "primitiveStatic.h"

using namespace VR;

void BerconMetaballPrimitiveStatic::getAverageResult(FieldAvg &field, Vector &pos) {	
	float radius = radii ? field.totalR/field.totalW : blobSize;
	if (field.totalW < blobSize)
		radius *= smoothstep1(field.totalW / trim, cutoff);
	field.res = radius - (pos - field.totalP / field.totalW).length();
}

void BerconMetaballPrimitiveStatic::getAverageResult(FieldGradAvg &grad) {	
	for (int i=0; i<4; i++) {
		float radius = radii ? grad.totalR[i]/grad.totalW[i] : blobSize;
		if (grad.totalW[i] < blobSize)
			radius *= smoothstep1(grad.totalW[i] / trim, cutoff);
		grad.res[i] = radius - (grad.location[i] - grad.totalP[i] / grad.totalW[i]).length();
	}
}

bool BerconMetaballPrimitiveStatic::intersectTraditional(Vector rO, Vector rD, BVHStep* first, bool insideField, Ireal stepLength, Ireal maxError, Ireal &dist) {
	BerconShadeContext sc;
	Ireal textureVal;
	BVHStep* cur;
	BVHStep* prev;
	BVHStep* step = first;		
	Ireal distLate;	
	if (insideField) { // Inside
		distLate = dist;
		while (step != NULL) {
			Vector pos = rO+rD*dist;
			if (tex) textureVal = getTexmap(sc, pos);
			Field field = Field(blobs, radii, pos, function, blobSize2);
			cur = step;
			prev = NULL;		
			while (cur != NULL && dist > cur->min) {
				// Optimize the ray BVHRay on the fly, however do it one step late because
				// we might need to use the step information when searching the exact hit.				
				if (distLate > cur->max) {
					if (cur == step)
						step = step->next;
					else {						
						prev->next = cur->next;
						cur->deleteThisOnly();
						cur = prev;
					}
					if (step == NULL) return false;
					cur = cur->next;
					continue;
				}				
				if (dist < cur->max) {
					cur->cont->getFieldNoTest(&field);
					Ireal fieldRes = field.res;
					if (tex) applyTexmap(textureVal, field.res);
					if (testThreshold(field.res, false))
						break;
					field.res = fieldRes; // Return to original value before applying texture
				}
				prev = cur;
				cur = cur->next;
			}
			if (tex) applyTexmap(textureVal, field.res);
			if (testThreshold(field.res, insideField))
				break;
			distLate = dist;
			dist += stepLength;
		}
	} else { // Outside
		if (dist < step->min)
			dist = step->min;
		distLate = dist;
		while (step != NULL) {		
			Vector pos = rO+rD*dist;
			Field field = Field(blobs, radii, pos, function, blobSize2);
			cur = step;
			prev = NULL;		
			while (cur != NULL && dist > cur->min) {
				// Optimize the ray BVHRay on the fly, however do it one step late because
				// we might need to use the step information when searching the exact hit.				
				if (distLate > cur->max) {
					if (cur == step)
						step = step->next;
					else {						
						prev->next = cur->next;
						cur->deleteThisOnly();
						cur = prev;
					}
					if (step == NULL) return false;
					cur = cur->next;
					continue;
				}				
				if (dist < cur->max)
					cur->cont->getFieldNoTest(&field);				
				prev = cur;
				cur = cur->next;
			}
			if (tex) applyTexmap(sc, pos, field.res);
			if (testThreshold(field.res, insideField))
				break;
			if (dist < step->min) {
				dist = step->min;
				distLate = dist;
			} else {
				distLate = dist;
				dist += stepLength;
			}
		}
	}
	
	if (step == NULL) return false; // No hit with the field

	// Confirmed hit, now we need to achieved desired maximum error	
	Ireal error = stepLength / 2.;
	dist -= error;		
	while (error > maxError) {
		Vector pos = rO+rD*dist;
		Field field = Field(blobs, radii, pos, function, blobSize2);
		cur = step;
		prev = NULL;		
		while (cur != NULL && dist > cur->min) {			
			if (dist < cur->max)
				cur->cont->getFieldNoTest(&field);
			prev = cur;
			cur = cur->next;
		}
		error /= 2.;		
		if (tex) applyTexmap(sc, pos, field.res);
		if (testThreshold(field.res, insideField))
			dist -= error;
		else
			dist += error;
	}
	return true;
}

bool BerconMetaballPrimitiveStatic::intersectAverage(Vector rO, Vector rD, BVHStep* first, bool insideField, Ireal stepLength, Ireal maxError, Ireal &dist) {
	BerconShadeContext sc;
	BVHStep* cur;
	BVHStep* prev;
	BVHStep* step = first;		
	Ireal distLate;	
	if (insideField) { // Inside
		distLate = dist;
		while (step != NULL) {
			Vector pos = rO+rD*dist;
			FieldAvg field = FieldAvg(blobs, radii, pos, function, blobSize2);
			cur = step;
			prev = NULL;		
			while (cur != NULL && dist > cur->min) {
				// Optimize the ray BVHRay on the fly, however do it one step late because
				// we might need to use the step information when searching the exact hit.				
				if (distLate > cur->max) {
					if (cur == step)
						step = step->next;
					else {						
						prev->next = cur->next;
						cur->deleteThisOnly();
						cur = prev;
					}
					if (step == NULL) return false;
					cur = cur->next;
					continue;
				}
				if (dist < cur->max)
					cur->cont->getFieldNoTest(&field);
				prev = cur;
				cur = cur->next;
			}
			getAverageResult(field, pos);
			if (tex) applyTexmap(sc, pos, field.res);
			if (testThreshold(field.res, insideField))
				break;
			distLate = dist;
			dist += stepLength;
		}
	} else { // Outside
		if (dist < step->min)
			dist = step->min;
		distLate = dist;
		while (step != NULL) {		
			Vector pos = rO+rD*dist;
			FieldAvg field = FieldAvg(blobs, radii, pos, function, blobSize2);
			cur = step;
			prev = NULL;		
			while (cur != NULL && dist > cur->min) {
				// Optimize the ray BVHRay on the fly, however do it one step late because
				// we might need to use the step information when searching the exact hit.				
				if (distLate > cur->max) {
					if (cur == step)
						step = step->next;
					else {						
						prev->next = cur->next;
						cur->deleteThisOnly();
						cur = prev;
					}
					if (step == NULL) return false;
					cur = cur->next;
					continue;
				}
				if (dist < cur->max)
					cur->cont->getFieldNoTest(&field);				
				prev = cur;
				cur = cur->next;
			}
			getAverageResult(field, pos);
			if (tex) applyTexmap(sc, pos, field.res);
			if (testThreshold(field.res, insideField))
				break;
			if (dist < step->min) {
				dist = step->min;
				distLate = dist;
			} else {
				distLate = dist;
				dist += stepLength;
			}
		}
	}
	
	if (step == NULL) return false; // No hit with the field

	// Confirmed hit, now we need to achieved desired maximum error	
	Ireal error = stepLength / 2.;
	dist -= error;		
	while (error > maxError) {
		Vector pos = rO+rD*dist;
		FieldAvg field = FieldAvg(blobs, radii, pos, function, blobSize2);
		cur = step;
		prev = NULL;		
		while (cur != NULL && dist > cur->min) {			
			if (dist < cur->max)
				cur->cont->getFieldNoTest(&field);
			prev = cur;
			cur = cur->next;
		}
		error /= 2.;		
		getAverageResult(field, pos);	
		if (tex) applyTexmap(sc, pos, field.res);
		if (testThreshold(field.res, insideField))
			dist -= error;
		else
			dist += error;
	}
	return true;
}

int BerconMetaballPrimitiveStatic::intersect(RSRay &ray) {	
	real dirLength = ray.dir.length();
	Ireal stepLength = this->stepLength / dirLength;
	Ireal maxError = this->maxError / dirLength;			
	Ireal dist = ray.cmint;
		  
	//if (ray.skipTag == this)
		dist += maxError * 2.;

	 // Infinite range, not cmint/cmaxt because it caused black "sliceplane" along x or y or x axis
	IRay tRay = IRay(ray.p, ray.dir);
	BVHRay bRay = BVHRay();
	bvh->traceRay(&tRay, &bRay);
	if (bRay.first == NULL) // No intersection with the bounding volume
		return false;	
	
	if (useAverage) {
		Vector pos = ray.p+ray.dir*dist;
		FieldAvg field = FieldAvg(blobs, radii, pos, function, blobSize2);
		bvh->getField(&field);	
		getAverageResult(field, pos);
		applyTexmap(pos, field.res);
		bool insideField = field.res > blobThreshold;	
		if (!intersectAverage(ray.p, ray.dir, bRay.first, insideField, stepLength, maxError, dist))
			return false;
	} else {
		Vector pos = ray.p+ray.dir*dist;
		Field field = Field(blobs, radii, pos, function, blobSize2);
		bvh->getField(&field);			
		applyTexmap(pos, field.res);
		bool insideField = field.res > blobThreshold;	
		if (!intersectTraditional(ray.p, ray.dir, bRay.first, insideField, stepLength, maxError, dist))
			return false;
	}
	
	if (dist>ray.is.t) return false;
	if (dist<=ray.cmint) return false;	
	if (dist>=ray.cmaxt) return false;

	MyCacheStruct cacheStruct;
	cacheStruct.rO = ray.p + ray.dir * dist;

	if (colors) { // For point weight calculations traditional and average are the same
		FieldColor field = FieldColor(blobs, radii, colors, cacheStruct.rO, function2, blobSize2); // <- function2
		bvh->getField(&field);
		cacheStruct.col = field.col / field.res;
	}

	ray.is.t = dist;
	ray.is.primitive = (GenericPrimitive*) this;
	ray.is.skipTag = this;
	
	raycache->putCache(*(VR::VRayContext*) ray.rayparams, cacheStruct);

	return true;
}

Vector BerconMetaballPrimitiveStatic::normal(Vector location) {
	Vector N;
	BerconShadeContext sc;
	if (useAverage) {
		FieldGradAvg grad = FieldGradAvg();
		grad.pointCloud = blobs; grad.radii = radii; grad.function = function; grad.size2 = blobSize2; 
		computeNormalLocations(grad.location, location);	
		bvh->getGradValue(&grad);			
		getAverageResult(grad);
		if (tex)
			for (int i=0; i<4; i++)
				applyTexmap(sc, grad.location[i], grad.res[i]);
		computeNormalResult(grad.res);
		N = -Vector(grad.res[1], grad.res[2], grad.res[3]);		
	} else {		
		FieldGrad grad = FieldGrad();
		grad.pointCloud = blobs; grad.radii = radii; grad.function = function; grad.size2 = blobSize2; 
		computeNormalLocations(grad.location, location);
		bvh->getGradValue(&grad);
		if (tex)
			for (int i=0; i<4; i++)
				applyTexmap(sc, grad.location[i], grad.res[i]);
		computeNormalResult(grad.res);
		N = -Vector(grad.res[1], grad.res[2], grad.res[3]);
	}
	return N;
}

Vector BerconMetaballPrimitiveStatic::getGNormal(RSRay &ray) {
	MyCacheStruct cacheStruct;
	raycache->getCache(*(VR::VRayContext*) ray.rayparams, cacheStruct);	
	return normal(cacheStruct.rO);
}

Vector BerconMetaballPrimitiveStatic::getGNormal(RSRayRef& rayref) {
	MyCacheStruct cacheStruct;
	raycache->getCache(*(VR::VRayContext*) rayref.getRayParams(), cacheStruct);			
	return normal(cacheStruct.rO);
}


void BerconMetaballPrimitiveStatic::getBBox(StaticBox &b) {
	b = bb;
}

void BerconMetaballPrimitiveStatic::split(int dim, real middle, StaticBox &bLeft, StaticBox &bRight) {
	bb.split(dim, middle, bLeft, bRight);
	return;
}

void BerconMetaballPrimitiveStatic::init(MetaballParams params, Vector* points, Ireal *sizes, Vector* colors, int pointCount, GeometryGenerator *owner, int ownerIndex, VR::RayCache<MyCacheStruct> *rayc) {
	setFunction(params.field, params.color);
	setParams(params);

	this->blobs = points;
	this->radii = sizes; // If radii is NULL then blobSize is used instead
	this->colors = colors;
	this->blobsCount = pointCount;

	bb.init();
	if (radii)
		for (int i=0;i<blobsCount;i++)
			bb.addExpanded(blobs[i], (real)radii[i]);
	else
		for (int i=0;i<blobsCount;i++)
			bb.addExpanded(blobs[i], (real)blobSize);

	this->bvh->create(this->blobs, this->radii, this->blobsCount, (real)this->blobSize, params.depth, params.leafSize, blobSize*params.leafLength);

	if (radii && !useAverage) // Square radii, we never need the normal radius after creating BVH, average mode requires the normal radius
		for (int i=0; i<this->blobsCount; i++)
			this->radii[i] *= this->radii[i];

	this->owner=owner;
	this->ownerIndex=ownerIndex;
	raycache=rayc;
}