/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#pragma once

#include "primitive.h"

/*
	Moving version of the primitive
*/
class BerconMetaballPrimitiveMoving: public BerconMetaballPrimitive, public VR::MovingPrimitive {
private:	
	/*
		Steps define number of samples taken in motion blur interval.		
		End state of step #S-1 equals starting state of #S so our arrays contain duplicate information.
		This is because its faster to compute position for point with starting point and velocity than interpolating
		between two points.

		P0 + time * (P1-P0)
		vs.
		P0 + time * VEL
		
		We don't have to calculate velocity every single time: (P1-P0) == VEL
		This means we use more ram but its faster to compute.
		Same goes for calculating changing radius.	
	*/
	VR::MovingBox bb;
	int* blobsCount;
	VR::Vector **blobs;	// pos, speed, pos, speed... etc.
	VR::Ireal **radii; // radii, change, radii, change.. etc.
	int steps;
	BoundingVolumeHierarchyMoving *bvh; // bvh, bvh... etc.
	VR::real *times; // For steps=3: 0 - 0.333 - 0.666 - 1.0 etc.
	VR::real timeInterval;

public:	
	
	BerconMetaballPrimitiveMoving() {
		blobs = NULL;
		radii = NULL;
		bvh = NULL;
		times = NULL;
		colors = NULL;
	}
	~BerconMetaballPrimitiveMoving() {
		for (int i=0; i<steps*2; i++) {
			if (blobs != NULL)
				delete[] (blobs[i]);
			if (radii != NULL)
				delete[] (radii[i]);		
		}
		delete[] blobs;
		delete[] radii;
		delete[] bvh;
		delete[] times;
		delete[] blobsCount;
		delete[] colors;
	}

	bool intersectTraditional(VR::Vector rO, VR::Vector rD, VR::real time, BVHMStep* &step, VR::Ireal stepLength, VR::Ireal maxError, VR::Ireal &dist, VR::Vector* pointCloud[2], VR::Ireal* sizes[2]);
	bool intersectAverage(VR::Vector rO, VR::Vector rD, VR::real time, BVHMStep* &step, VR::Ireal stepLength, VR::Ireal maxError, VR::Ireal &dist, VR::Vector* pointCloud[2], VR::Ireal* sizes[2]);

	inline void getField(BVHMStep* cur, VR::real time, FieldMoving* field, VR::Vector* pointCloud[2], VR::Ireal* sizes[2], bool linear);
	inline void getAverageResult(FieldMovingAvg &field, VR::Vector &pos);
	inline void getAverageResult(FieldGradMovingAvg &grad);

	/* VRay interface */
	int expandable(void) { return false; }
	bool splittable(void) { return true; }	
	int intersect(VR::RSRay &ray);
	virtual VR::Vector getGNormal(VR::RSRay &ray);
	virtual VR::Vector getGNormal(VR::RSRayRef& rayref);
	int getMemUsage(void) { return 0; }

	uint32 type(void) { return BERCON_METABALL_PRIM_STATIC; }
	void getBBox(VR::MovingBox &bbox) {bbox=bb;}
	void split(int dim, VR::real middle, VR::MovingBox &bLeft, VR::MovingBox &bRight) {bb.split(dim, middle, bLeft, bRight);}		
	void init(MetaballParams params, VR::Vector** points, VR::Ireal** sizes, VR::Vector* colors, int* pointCount, VR::GeometryGenerator *owner, int ownerIndex, VR::RayCache<MyCacheStruct> *rayc);
};