/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#pragma once

#include "vraybase.h"

/*
	These classes do the actual field computation.

	Field* class is passed around and each BVH container adds its points to it with addPoints()

	This way we can implement any kind of technique to compute the field.

	Good example of this is the average field method where weighted average of all the points locations
	is used.
*/


/*
	Normal field value computations, field = sum of all metaballs
*/

// STATIC

class Field {
public:	
	VR::Ireal res;
	VR::Vector location;
	VR::Ireal size2;	
	VR::Ireal (*function)(VR::Ireal, VR::Ireal);
	VR::Vector* pointCloud;	
	VR::Ireal* radii;
	
	virtual void addPoints(int* points, int count);
	Field() {}
	Field(VR::Vector* pointCloud, VR::Ireal* radii, VR::Vector location, VR::Ireal (*function)(VR::Ireal, VR::Ireal), VR::real size2) {
		this->pointCloud=pointCloud; this->radii=radii; this->location=location; this->function=function; this->size2=size2; res=0.;
	}
	~Field() {}
};

class FieldColor: public Field {
public:
	VR::Vector* colors;
	VR::Vector col;

	void addPoints(int* points, int count);
	FieldColor() {}
	FieldColor(VR::Vector* pointCloud, VR::Ireal* radii, VR::Vector* colors, VR::Vector location, VR::Ireal (*function)(VR::Ireal, VR::Ireal), VR::real size2) {
		this->pointCloud=pointCloud; this->radii=radii; this->location=location; this->function=function; this->size2=size2; res=0.;
		this->colors=colors; col.makeZero();
	}
	~FieldColor() {}
};

class FieldGrad {
public:
	VR::Ireal res[4];
	VR::Vector location[4];
	VR::Ireal size2;	
	VR::Ireal (*function)(VR::Ireal, VR::Ireal);
	VR::Vector* pointCloud;
	VR::Ireal* radii;

	virtual void addPoints(int* points, int count);
	FieldGrad() {for (int i=0;i<4;i++) res[i] = 0.;}
	~FieldGrad() {}
};

// MOVING

class FieldMoving {
public:	
	VR::Ireal res;
	VR::Vector location;
	VR::Ireal size2;	
	VR::Ireal (*function)(VR::Ireal, VR::Ireal);
	
	virtual void addPoints(VR::Vector* cloud, int count);
	virtual void addPoints(VR::Vector* cloud, VR::Ireal* radii, VR::Ireal* sizesLin, int count);
	FieldMoving() {}
	FieldMoving(VR::Vector location, VR::Ireal (*function)(VR::Ireal, VR::Ireal), VR::real size2) {
		this->location=location; this->function=function; this->size2=size2; res=0.;		
	}	
	~FieldMoving() {}
};

class FieldGradMoving {
public:
	VR::Vector* colors;
	VR::Vector col;
	VR::Ireal colr;

	VR::Ireal res[4];
	VR::Vector location[4];
	VR::Ireal size2;	
	VR::Ireal (*function)(VR::Ireal, VR::Ireal);
	VR::Ireal (*function2)(VR::Ireal, VR::Ireal);

	//virtual void addPoints(VR::Vector* cloud, int count);
	//virtual void addPoints(VR::Vector* cloud, VR::Ireal* radii, VR::Ireal* sizesLin, int count);
	virtual void addPoints(VR::Vector* cloud, int* points, int count);
	virtual void addPoints(VR::Vector* cloud, VR::Ireal* radii, VR::Ireal* sizesLin, int* points, int count);
	FieldGradMoving() {}
	FieldGradMoving(VR::Vector* colors) {for (int i=0;i<4;i++) res[i] = 0.; this->colors = colors; col.makeZero(); colr = 0.; }
	~FieldGradMoving() {}
};

/*
	Average field value computations, compute average position for metaballs and compute its field
*/

// STATIC

class FieldAvg: public Field {
public:	
	VR::Vector totalP;
	VR::Ireal totalW;
	VR::Ireal totalR;
	void addPoints(int* points, int count);
	FieldAvg(VR::Vector* pointCloud, VR::Ireal* radii, VR::Vector location, VR::Ireal (*function)(VR::Ireal, VR::Ireal), VR::real size2) {
		this->pointCloud=pointCloud; this->radii=radii; this->location=location; this->function=function;
		this->size2=size2; res=0.; totalW=0.; totalR=0.; totalP=VR::Vector(0,0,0);
	}
};

class FieldGradAvg: public FieldGrad {
public:
	VR::Vector totalP[4];
	VR::Ireal totalW[4];
	VR::Ireal totalR[4];
	void addPoints(int* points, int count);
	FieldGradAvg() {
		for (int i=0;i<4;i++) {res[i] = 0.; totalW[i]=0.; totalR[i]=0.; totalP[i]=VR::Vector(0,0,0);}
	}
};

// MOVING

class FieldMovingAvg: public FieldMoving {
public:	
	VR::Vector totalP;
	VR::Ireal totalW;	
	VR::Ireal totalR;
	void addPoints(VR::Vector* cloud, int count);
	void addPoints(VR::Vector* cloud, VR::Ireal* radii, VR::Ireal* sizesLin, int count);
	FieldMovingAvg(VR::Vector location, VR::Ireal (*function)(VR::Ireal, VR::Ireal), VR::real size2) {
		this->location=location; this->function=function; this->size2=size2; res=0.; totalW=0.; totalR=0.; totalP=VR::Vector(0,0,0);
	}
};

class FieldGradMovingAvg: public FieldGradMoving {
public:
	VR::Vector totalP[4];
	VR::Ireal totalW[4];
	VR::Ireal totalR[4];
	void addPoints(VR::Vector* cloud, int* points, int count);
	void addPoints(VR::Vector* cloud, VR::Ireal* radii, VR::Ireal* sizesLin, int* points, int count);
	FieldGradMovingAvg(VR::Vector* colors) {
		for (int i=0;i<4;i++) {res[i] = 0.; totalW[i]=0.; totalR[i]=0.; totalP[i]=VR::Vector(0,0,0);  this->colors = colors; col.makeZero(); colr = 0.; }
	}
};