/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#include "max.h"
#include "BVHStatic.h"

using namespace VR;

static void bAddExpanded(Box &b, const Vector &p, real dist) {
	for (int i=0; i<3; i++) {
		if (p[i]-dist<b.pmin[i]) b.pmin[i]=p[i]-dist;
		if (p[i]+dist>b.pmax[i]) b.pmax[i]=p[i]+dist;
	}
}

/*
	############ BVHRay ############
*/

void BVHRay::add(VR::Ireal min, VR::Ireal max, BVHContainer* cont) {
	steps.push_back(new BVHStep(min, max, cont, NULL));
	
	/*
	if (this->first == NULL) {
		this->first = new BVHStep(min, max, cont, NULL);
		return;
	}

	if (min < this->first->min) {
		this->first = new BVHStep(min, max, cont, this->first);
		return;
	}

	BVHStep* cur = this->first;
	while (cur->next != NULL && min > cur->next->min)
		cur = cur->next;
	cur->next = new BVHStep(min, max, cont, cur->next);
	*/
}

void BVHRay::sortSteps() {
	std::sort(steps.begin(), steps.end(), BVHStepSort());	
}

void BVHRay::collapseSteps() {
	if (steps.size() == 0) return;
	
	this->first = steps[0];

	for (int i=1; i<steps.size(); i++)
		steps[i-1]->next = steps[i];

	steps.clear();
}

/*
	############ BVHNode ############
*/

int BVHNode::intersect(VR::IRay* ray) {
	if (ray->testIntersectBox(this->b)) {
		if (!this->type())
			return true;
		if (this->getLeft()->intersect(ray))
			return true;
		if (this->getRight()->intersect(ray))
			return true;
	}	
	return false;
}

/*
	############ Branch ############
*/

void Branch::traceRay(VR::IRay* ray, BVHRay* bRay) {
	if (ray->testIntersectBox(this->b)) {
		this->getLeft()->traceRay(ray, bRay);
		this->getRight()->traceRay(ray, bRay);
	}
}

void Branch::getField(Field* field) {
	if (b.isInside(field->location)) {
		this->getLeft()->getField(field);
		this->getRight()->getField(field);
	}
}

void Branch::getGradValue(FieldGrad* grad) {
	if (b.isInside(grad->location[0])) {
		this->getLeft()->getGradValue(grad);
		this->getRight()->getGradValue(grad);
	}
}

Branch::Branch(VR::Vector* pointCloud, Ireal* radii, int* points, int count, real size, int maxDepth, int depth, int leafSize, real maxLength) {
	// Create bounding box
	b.init();	
	for (int i=0;i<count;i++)
		if (radii)
			bAddExpanded(b, pointCloud[points[i]], radii[points[i]]);
		else
			bAddExpanded(b, pointCloud[points[i]], size);

	// Choose which axis to split and where
	int axis = b.maxDimension();
	real middle = (b.pmin[axis] + b.pmax[axis]) * 0.5f;
	
	// Count number of points on both sides
	int leftCount = 0;
	for (int i=0;i<count;i++)
		if (pointCloud[points[i]][axis] < middle)
			leftCount++;
	int rightCount = count - leftCount;

	// If the other side gets 0 points, all the points are at the same location
	// and there is no longer point dividing them any futher so we just make
	// leaves here
	bool forceLeaf = false;
	if (leftCount == 0 || rightCount == 0) {
		leftCount = count / 2;
		rightCount = count - leftCount;
		forceLeaf = true;
	}

	// Create fitting arrays for both
	int* leftPoints;
	int* rightPoints;	
	//if (leftCount > 0)
		leftPoints = new int[leftCount];
	//if (rightCount > 0)
		rightPoints = new int[rightCount];
	
	// Assign points to for both sides
	if (forceLeaf) {		
		int j = 0;
		for (int i=0;i<count;i++)
			if (i<leftCount)
				leftPoints[i] = points[i];
			else
				rightPoints[j++] = points[i];

	} else {
		int j = 0, k = 0;
		for (int i=0;i<count;i++)
			if (pointCloud[points[i]][axis] < middle) {
				//if (j < leftCount)
					leftPoints[j++] = points[i];
			} else {
				//if (k < rightCount)
					rightPoints[k++] = points[i];
			}
	}

	// Free previous point array, as its now split into left and right
	delete[] points;

	// Leaves or not?
	real length = b.pmax[axis] - b.pmin[axis];
	
	if (maxDepth <= depth || ((length < maxLength || radii) && leftCount < leafSize) || forceLeaf) // Make leaf
		this->left = new Leaf(pointCloud, radii, leftPoints, leftCount, size);
	else // Branch more		
		this->left = new Branch(pointCloud, radii, leftPoints, leftCount, size, maxDepth, depth+1, leafSize, maxLength);

	if (maxDepth <= depth || ((length < maxLength || radii) && rightCount < leafSize) || forceLeaf) // Make leaf
		this->right = new Leaf(pointCloud, radii, rightPoints, rightCount, size);
	else // Branch more		
		this->right = new Branch(pointCloud, radii, rightPoints, rightCount, size, maxDepth, depth+1, leafSize, maxLength);
}

/*
	############ Leaf ############
*/

void Leaf::traceRay(VR::IRay* ray, BVHRay* bRay) {
	IRay ray2 = IRay(*ray);
	if (ray2.intersectBox(this->b))
		bRay->add(ray2.cmint, ray2.cmaxt, this);			
}

void Leaf::getField(Field* field) {
	if (b.isInside(field->location))
		field->addPoints(points, count);	
}

void Leaf::getFieldNoTest(Field* field) {
	field->addPoints(points, count);
}

void Leaf::getGradValue(FieldGrad* grad) {
	if (b.isInside(grad->location[0]))
		grad->addPoints(points, count);		
}

Leaf::Leaf(VR::Vector* pointCloud, Ireal* radii, int* points, int count, real size) {
	b.init();	
	for (int i=0;i<count;i++)
		if (radii)
			bAddExpanded(b, pointCloud[points[i]], radii[points[i]]);
		else
			bAddExpanded(b, pointCloud[points[i]], size);						
	this->points = points;
	this->count = count;		
}

/*
	############ BoundingVolumeHierarchy ############
*/

void BoundingVolumeHierarchy::create(VR::Vector* pointCloud, Ireal* radii, int count, real size, int maxDepth, int leafSize, real maxLength) {
	delete this->root;
	int* points = new int[count];
	for (int i=0;i<count;i++)
		points[i] = i;
	this->root = new Branch(pointCloud, radii, points, count, size, maxDepth, 1, leafSize, maxLength);
}