/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#pragma once

#include "vraybase.h"
#include "vraygeom.h"
#include "rayserver.h"

#include "vrayrenderer.h"
#include "vrayplugins.h"
#include "vraytexutils.h"

#include "field.h"

#include <algorithm>
#include <vector>

class BVHMLeaf;

/*
	When leaf is traced its points positions are cached into this container so
	they don't have to recalculated for this ray again.
	For each iteration step we save (POSITION + VELOCITY * TIME) * POINTCOUNT
*/
class BVHMContainer {
public:	
	VR::Vector* cloud;
	VR::Ireal* sizes;
	VR::Ireal* sizesLin;
	int count;
	bool noPositions;
	BVHMLeaf* leaf;

	BVHMContainer(BVHMLeaf* leaf) {
		this->leaf = leaf;
		cloud = NULL;
		sizes = NULL;
		sizesLin = NULL;
		noPositions = true;
	}
	~BVHMContainer() {
		delete cloud;
		delete sizes;
		delete sizesLin;
	}
	
	void calculatePositions(VR::Vector* pointCloud[2], VR::real time);	
	void calculatePositions(VR::Vector* pointCloud[2], VR::Ireal* radii[2], VR::real time, bool linear);	
	void getField(VR::Vector* pointCloud[2], VR::real time, FieldMoving* field);
	//void getFieldGrad(VR::Vector* pointCloud[2], VR::real time, FieldGradMoving* grad);
	void getField(VR::Vector* pointCloud[2], VR::Ireal* radii[2], VR::real time, FieldMoving* field, bool linear);
	//void getFieldGrad(VR::Vector* pointCloud[2], VR::Ireal* radii[2], VR::real time, FieldGradMoving* grad, bool linear);

	void getFieldGrad(VR::Vector* pointCloud[2], VR::real time, FieldGradMoving* grad);
	void getFieldGrad(VR::Vector* pointCloud[2], VR::Ireal* radii[2], VR::real time, FieldGradMoving* grad, bool linear);
};

/*
	When we trace the BVH each leaf thats along the ray is turned into these
	They contain the positions along the ray where it enters the bounding box
	of the leaf and when it exits it. This means the BVH has to be intersected
	for each ray only once.
*/
class BVHMStep {
public:
	VR::Ireal min;
	VR::Ireal max;
	BVHMContainer* cont;
	BVHMStep* next; // This may be change to skip steps when marching
	
	BVHMStep(VR::Ireal min, VR::Ireal max, BVHMContainer* cont, BVHMStep* next) {
		this->min=min; this->max=max; this->cont=cont; this->next=next;
	}
	~BVHMStep() {delete cont; delete next;}
	void deleteThisOnly() {next = NULL; delete this;} // Used when optimizing, sometimes we want to remove individual steps
};
struct BVHMStepSort {
     bool operator()(BVHMStep*& a, BVHMStep*& b) {
          return a->min < b->min;
     }
};

/*
	Ray structure that contains all the leafs along single ray
*/
class BVHMRay {
public:
	BVHMStep* first;
	std::vector<BVHMStep*> steps;

	void add(VR::Ireal min, VR::Ireal max, BVHMContainer* cont);
	void sortSteps();
	void collapseSteps();

	
	BVHMRay() {first = NULL; steps.reserve(25); }
	~BVHMRay() {delete first;}
};

/*
	Parent class for BVH nodes
*/
class BVHMNode {
public:
	VR::StaticBox b[2];
		
	BVHMNode() {}
	~BVHMNode() {}

	int intersect(VR::IRay* ray, VR::real time);	
	void initBoundingBox(VR::Vector* pointCloud[2], VR::Ireal* radii[2], int* points, int count, VR::real size);

	// Override
	virtual int type() {return 0;}
	virtual void traceRay(VR::IRay* ray, BVHMRay* bRay, VR::real time) = 0;	

	// For branches
	virtual BVHMNode* getLeft() {return NULL;}
	virtual BVHMNode* getRight() {return NULL;}

	// For leaves
	virtual int* getPoints() {return NULL;}	
	virtual int getCount() {return 0;}
};

/*
	BVH branch class
*/
class BVHMBranch: public BVHMNode {
public:
	BVHMNode* left;
	BVHMNode* right;

	int type() {return 1;}			
	void traceRay(VR::IRay* ray, BVHMRay* bRay, VR::real time);
	
	BVHMNode* getLeft() {return left;}
	BVHMNode* getRight() {return right;}

	BVHMBranch(VR::Vector* pointCloud[2], VR::Ireal* radii[2], int* points, int count, VR::real size, int maxDepth, int depth, int leafSize, VR::real maxLength);	
	~BVHMBranch() {
		delete left;
		delete right;
	}
};

/*
	BVH leaf class, leafs only store indexes of points, they don't
	store the actual points. This makes the structure more flexible
	and we might extend the number position samples in single frame
	in the future more easily.
*/
class BVHMLeaf: public BVHMNode {
public:
	int* points;
	int count;

	int type() {return 0;}	
	void traceRay(VR::IRay* ray, BVHMRay* bRay, VR::real time);

	int* getPoints() {return points;}
	int getCount() {return count;}	
	
	BVHMLeaf(VR::Vector* pointCloud[2], VR::Ireal* radii[2], int* points, int count, VR::real size);	
	~BVHMLeaf() {
		delete[] points;
	}
};

/*
	BVH base class
*/
class BoundingVolumeHierarchyMoving {
private:
	BVHMNode* root;
	
public:
	BoundingVolumeHierarchyMoving() {
		root = NULL;
	}

	~BoundingVolumeHierarchyMoving() {
		delete root;
	}

	void create(VR::Vector* pointCloud[2], VR::Ireal* radii[2], int count, VR::real size, int maxDepth, int leafSize, VR::real maxLength);

	int intersect(VR::IRay* ray, VR::real time) {	
		return root->intersect(ray, time);
	}
	void traceRay(VR::IRay* ray, BVHMRay* bRay, VR::real time) {
		root->traceRay(ray, bRay, time);
		bRay->sortSteps();
		bRay->collapseSteps();
	}
};