-- Lightning script for 3dsmax 8
-- Author: Jerry Ylilammi


-- #############################################################################################
-- #################################/ Global Parameters        \################################
-- #############################################################################################

	global splineDetail
	global splineRender
	global splineDisplay
	global splineRadius
	global splineSides

	global noiseFractInterval = 0.5
	global noiseLacunarity = 1.75
	global noiseOctaves = 4
	global noiseStrength
	global noiseSize
	global noiseWhiteStrength
	global noiseSeed

	global forkLocationMin
	global forkLocationMax
	global forkLengthMin
	global forkLengthMax
	global forkAngleMin
	global forkAngleMax
	global forkAmount
	global forkIterations
	global forkBranchesMin
	global forkBranchesMax
	global forkBranchLengthMin
	global forkBranchLengthMax

	global startTime
	global endTime
	global animateLightning
	global animateNoise
	global animationSpeed
	global keyInterval

	global sourceObj = undefined
	global targetObjs = #()
	
	global lightningCS
	global lightningLength
	global currentKnot
	global fadeArray

-- #############################################################################################
-- #################################/ Functions                \################################
-- #############################################################################################

-- #################### // selectByName Filer  \\ ####################
fn sourceExcludeFilter obj = (
	if sourceObj == obj then false
	else if (findItem targetObjs obj) != 0 then false
	else true
)

fn getFade current = (
	if splineDetail >= current then (
		(sin((current-1.0)/(splineDetail-1.0)*180))
	) else (
		1
	)
)

-- #################### // Create spline \\ ####################
fn createNewSpline spline pointA pointB knots rootPoint = (
	addNewSpline spline
	addKnot spline (numSplines spline) #corner #line pointA
	fadeArray[currentKnot] = getFade rootPoint
	currentKnot += 1
  for i in 1 to (knots-1) do (
		addKnot spline (numSplines spline) #corner #line ( pointA + ( (pointB-pointA) * (1/(knots-1 as float)*i) ) )
		fadeArray[currentKnot] = getFade (i+rootPoint)
		currentKnot += 1
  )
	updateShape spline
)

-- #################### // Create fork \\ ####################
fn createFork spline index rootLength rootPoint iterLevel = (
	knots = numKnots spline index

	-- random fork source
	sourceIndex = random (forkLocationMin*knots as integer) (forkLocationMax*knots as integer)
	if sourceIndex < 1 then sourceIndex = 1
	forkSource = getKnotPoint spline index sourceIndex

	-- random fork target
	if iterLevel == 0 then
		lengthMultiplier = random forkLengthMin forkLengthMax
	else
		lengthMultiplier = random forkBranchLengthMin forkBranchLengthMax

	forkLength = rootLength * lengthMultiplier
	randomRotation = random -180.0 180.0
	randomAngle = random (forkAngleMin*180) (forkAngleMax*180)
	distanceNormal = sin(randomAngle) * forkLength
	distanceTangent = cos(randomAngle) * forkLength

	-- fork target in lightningCS
	forkTarget = [(cos(randomRotation)*distanceNormal), (sin(randomRotation)*distanceNormal), distanceTangent]

	-- fork target to world
	forkTarget = lightningCS[1] * forkTarget[1] + lightningCS[2] * forkTarget[2] + lightningCS[3] * forkTarget[3] + lightningCS[4]
	
	-- offset
	forkTarget = forkTarget + forkSource
  
	-- calculate detail based on length compared to main lightning
	forkDetail = rootLength / lightningLength * lengthMultiplier * splineDetail

	-- create fork spline
	createNewSpline spline forkSource forkTarget (forkDetail as integer) (sourceIndex+rootPoint)

	-- create branches
	if iterLevel < forkIterations then (
		rootIndex = numSplines spline
		branchCount = random forkBranchesMin forkBranchesMax
		i = 1
		while i <= branchCount do (
		 	createFork spline rootIndex forkLength (sourceIndex+rootPoint) (iterLevel+1)
		 	i += 1
		)
	)
)

-- #################### // Create lightning \\ ####################
fn createLightningStrike sourcePoint targetPoint progBar sTime eTime = (
		 	lightning = SplineShape()
		 	fadeArray = #()
		 	currentKnot = 1
		 	lightningLength = length (targetPoint - sourcePoint)
		 	progBar.value = 0

			-- #################### // Create main lightning spline and forks \\ ####################

			-- create lightning main spline and global lightning coordinate system
		 	createNewSpline lightning sourcePoint targetPoint splineDetail 0
			lightningCS = matrixFromNormal (normalize (targetPoint - sourcePoint))

			-- update fadeArray (main spline)
      for i in 1 to splineDetail do (
				fadeArray[i] = getFade i
			)

			-- create forks
			for i in 2 to (forkAmount+1) do (
      	createFork lightning 1 lightningLength 0 0
			)

			updateShape lightning

			-- #################### // Animate noise \\ ####################

			lightningProxy = lightning -- Need duplicate to store knot positions to which noise is then applied
      if animateNoise then animateVertex lightning #all
			exitLoop = false

			-- for each frame
			time = sTime
			while time <= eTime and exitLoop == false do (
				knotIndex = 1
		  	-- for each spline
		  	index = 1
		  	while index <= (numSplines lightning) do (

					-- for each knot
				  k = numKnots lightning index
					for i in 1 to k do (
					  knotProxy = knotPosition = getKnotPoint lightningProxy index i
					  phase = time*animationSpeed

						-- calculate noise strength
					  strength = fadeArray[knotIndex] * noiseStrength

						-- calculate large noise, position from proxy (static)
						knotPosition.x = knotProxy.x + (fractalNoise [knotProxy.y*noiseSize, phase, noiseSeed] noiseFractInterval noiseLacunarity noiseOctaves) * strength
						knotPosition.y = knotProxy.y + (fractalNoise [noiseSeed, knotProxy.z*noiseSize, phase] noiseFractInterval noiseLacunarity noiseOctaves) * strength
						knotPosition.z = knotProxy.z + (fractalNoise [phase, noiseSeed, knotProxy.x*noiseSize] noiseFractInterval noiseLacunarity noiseOctaves) * strength

						-- calculate small noise, position from knot (dynamic)
						knotPosition.x = knotPosition.x + (noise3 [knotPosition.y*noiseSize*1000.0, phase*2, noiseSeed]) * strength * noiseWhiteStrength
						knotPosition.y = knotPosition.y + (noise3 [noiseSeed, knotPosition.z*noiseSize*1000.0, phase*2]) * strength * noiseWhiteStrength
						knotPosition.z = knotPosition.z + (noise3 [phase*2, noiseSeed, knotPosition.x*noiseSize*1000.0]) * strength * noiseWhiteStrength

						-- add key / set vertex position
						if animateNoise then (
	            trackIndex = 3*(knotIndex-1)+2
				   		key = addNewKey lightning[#Object__Editable_Spline][#master][trackIndex].controller time
							key.value = knotPosition
						) else (
							setKnotPoint lightning index i knotPosition
							exitLoop = true
						)
						knotIndex += 1
					)
					index += 1
				)

				-- update progress bar
				progBar.value = ((time as float)/(eTime-sTime))*100 as integer
				time = time + keyInterval
			)

			-- Set pivot & color & display
			lightning.wireColor = white
 			lightning.pivot = lightning.center
			lightning.steps = 0
			lightning.render_renderable = splineRender
			lightning.render_displayRenderMesh = splineDisplay
			lightning.render_thickness = splineRadius
      lightning.render_sides = splineSides
			updateShape lightning

 			
 			-- Animate visibility
 			if animateLightning == true then (
	 			lightning.visibility = bezier_float()
	 			key = addNewKey lightning.visibility.controller 0
				if sTime == 0 then (
					key.value = 1.0
				)	else (
					key.value = 0.0
		 			key = addNewKey lightning.visibility.controller (sTime-1)
					key.value = 0.0
		 			key = addNewKey lightning.visibility.controller sTime
					key.value = 1.0
				)
				key = addNewKey lightning.visibility.controller eTime
				key.value = 1.0
				if animationRange.end != eTime then (
					key = addNewKey lightning.visibility.controller (eTime+1)
					key.value = 0.0
				)
			)
			
		 	progBar.value = 0
)

-- #############################################################################################
-- #################################/ User Interface           \################################
-- #############################################################################################

rollout lightning_Spline "Spline Parameters" (
	spinner spinnerDetail "Vertex count:" range:[0,10000,150] type:#integer offset:[9,0]
	checkbox checkboxRenderable "Renderable" checked:true offset:[-9,0]
	checkbox checkboxDisplay "Display Render Mesh" offset:[-9,0]
	spinner spinnerRadius "Radius:" range:[0.0, 100.0, 0.5] type:#float offset:[9,0]
	spinner spinnerSides "Sides:" range:[0, 100, 8] type:#integer offset:[9,0]
)

rollout lightning_Forks "Fork Parameters" rolledUp:true (
	checkbox checkboxEnableForks "Enable forks" offset:[-9,0] checked:true

	spinner spinnerForks "Amount:" range:[0,10000,7] type:#integer offset:[9,0]
	group "Location" (
		spinner spinnerForksLocationMin "Start:" range:[0.0, 1.0, 0.05] type:#float offset:[4,0]
		spinner spinnerForksLocationMax "End:" range:[0.0, 1.0, 0.7] type:#float offset:[4,0]
	)
	group "Length" (
		spinner spinnerForksLengthMin "Min:" range:[0.0, 1.0, 0.15] type:#float offset:[4,0]
		spinner spinnerForksLengthMax "Max:" range:[0.0, 1.0, 0.4] type:#float offset:[4,0]
	)
	group "Angle" (
		spinner spinnerForksAngleMin "Min:" range:[0.0, 1.0, 0.1] type:#float offset:[4,0]
		spinner spinnerForksAngleMax "Max:" range:[0.0, 1.0, 0.4] type:#float offset:[4,0]
	)
	group "Branches" (
		spinner spinnerBranchesMin "Min Amount:" range:[0, 100, 0] type:#integer offset:[4,0]
		spinner spinnerBranchesMax "Max Amount:" range:[0, 100, 2] type:#integer offset:[4,0]
		spinner spinnerBranchLengthMin "Min Length:" range:[0.0, 1.0, 0.5] type:#float offset:[4,0]
		spinner spinnerBranchLengthMax "Max Length:" range:[0.0, 1.0, 0.8] type:#float offset:[4,0]
		spinner spinnerIterations "Iterations:" range:[0,10,2] type:#integer offset:[4,0]
	)

	on checkboxEnableForks changed checkboxState do (
  	spinnerForks.enabled = checkboxState
		spinnerForksLocationMin.enabled = checkboxState
		spinnerForksLocationMax.enabled = checkboxState
		spinnerForksLengthMin.enabled = checkboxState
		spinnerForksLengthMax.enabled = checkboxState
		spinnerForksAngleMin.enabled = checkboxState
		spinnerForksAngleMax.enabled = checkboxState
		spinnerBranchesMin.enabled = checkboxState
		spinnerBranchesMax.enabled = checkboxState
		spinnerBranchLengthMin.enabled = checkboxState
		spinnerBranchLengthMax.enabled = checkboxState
		spinnerIterations.enabled = checkboxState
	)
)

rollout lightning_Noise "Noise Parameters" rolledUp:true (
	spinner spinnerNoiseSeed "Seed:" range:[0, 10000, (random 0 10000)] type:#integer offset:[9,0]
	spinner spinnerNoiseSize "Size:" range:[0.0, 10000.0, 0.05] type:#float offset:[9,0]
	spinner spinnerNoiseStrength "Strength:" range:[0.0, 10000.0, 10.0] type:#float offset:[9,0]
	spinner spinnerNoiseWhite "Whitenoise:" range:[0.0, 1.0, 0.2] type:#float offset:[9,0]
)

rollout lightning_Animation "Animation Parameters" rolledUp:true (
	checkbox checkboxAnimateLightning "Animate lightning" offset:[-9,0]
	spinner spinnerStartTime "Start time:" range:[0,10000,0] type:#integer offset:[9,0] enabled:false
	spinner spinnerEndTime "End time:" range:[0,10000,30] type:#integer offset:[9,0]enabled:false

	checkbox checkboxStrikesPerObject "Strikes per object" offset:[-9,0] enabled:false
	spinner spinnerSrikeChance "Chance to srike:" range:[0.0,1000.0,0.2] type:#float fieldwidth:30 offset:[9,0] enabled:false
	spinner spinnerSimStrikes "Simultaneous strikes:" range:[0,1000,4] type:#integer fieldwidth:30 offset:[9,0] enabled:false

	group "Strike duration" (
		spinner spinnerStrikeDurMin "Min:" range:[0,10000,3] type:#integer offset:[4,0] enabled:false
		spinner spinnerStrikeDurMax "Max:" range:[0,10000,6] type:#integer offset:[4,0] enabled:false
	)

	checkbox checkboxAnimateNoise "Animate noise" offset:[-9,0] enabled:false
	spinner spinnerSpeed "Noise speed:" range:[0.0,10000.0,0.3] type:#float offset:[9,0] enabled:false
	spinner spinnerKeys "Frames per key:" range:[1,10,1] type:#integer offset:[9,0] enabled:false
	
	on checkboxAnimateLightning changed checkboxState do (
  	spinnerStartTime.enabled = checkboxState
		spinnerEndTime.enabled = checkboxState
		checkboxStrikesPerObject.enabled = checkboxState
		spinnerSrikeChance.enabled = checkboxState
		spinnerSimStrikes.enabled = checkboxState
		spinnerStrikeDurMin.enabled = checkboxState
		spinnerStrikeDurMax.enabled = checkboxState
		checkboxAnimateNoise.enabled = checkboxState
		spinnerSpeed.enabled = checkboxState
		spinnerKeys.enabled = checkboxState
	)

)

-- #################### // Main Parameters UI \\ ####################

rollout lightning_Main "Main Parameters" (

	group "Source" (
		dropdownlist dropdownSourceType width:132 items:#("Pivot", "Selected Verties", "Verties", "Selected Faces", "Faces") offset:[-3,0]
  	pickbutton buttonSource "Source Object" width:130 autoDisplay:true filter:sourceExcludeFilter
	)

	group "Targets" (
		dropdownlist dropdownTargetType width:132 items:#("Pivot", "Selected Verties", "Verties", "Selected Faces", "Faces") offset:[-3,0]
    listbox listBoxTargets height:6 width:130 offset:[-4,0]
    pickbutton buttonTargetAdd "Add" width:30 height:18 filter:sourceExcludeFilter across:3 offset:[-7,0]
    button buttonTargetByList "By List" width:40 height:18 offset:[-7,0]
    button buttonTargetRemove "Remove" width:45 height:18 offset:[7,0]
	)
	
	progressbar progBar width:142 height:20 offset:[-11,0] color:blue
 	button buttonCreate "Create lightning" width:142 offset:[-2,0]

  -- #################### // Main Parameters Actions \\ ####################

	on buttonSource picked obj do	(
 		sourceObj = obj
 	)

 	on buttonTargetAdd picked obj do (
 		append targetObjs obj
		targetList = listBoxTargets.items
 		append targetList obj.name
    listBoxTargets.items = targetList
 	)

 	on buttonTargetByList pressed do (
		selectionArray = selectByName title:"Select source objects" filter:sourceExcludeFilter
		if selectionArray != undefined then (
			join targetObjs selectionArray
			targetList = listBoxTargets.items
			for obj in selectionArray do (
	 			append targetList obj.name
			)
	    listBoxTargets.items = targetList
	  )
	)

 	on buttonTargetRemove pressed do (
 		deleteItem targetObjs listBoxTargets.selection
		targetList = listBoxTargets.items
 		deleteItem targetList listBoxTargets.selection
    listBoxTargets.items = targetList
 	)

  -- #################### // Start lightning creation \\ ####################

 	on buttonCreate pressed do (
 		if sourceObj == undefined then (
     		messagebox "You must choose source object"
    ) else if targetObjs[1] == undefined then (
		 		messagebox "You must choose target object"
		) else (

			-- Global values
  		splineDetail = lightning_Spline.spinnerDetail.value
      splineRender = lightning_Spline.checkboxRenderable.checked
      splineDisplay = lightning_Spline.checkboxDisplay.checked
      splineRadius = lightning_Spline.spinnerRadius.value
      splineSides = lightning_Spline.spinnerSides.value

  		noiseSize = lightning_Noise.spinnerNoiseSize.value
			noiseStrength = lightning_Noise.spinnerNoiseStrength.value
      noiseWhiteStrength = lightning_Noise.spinnerNoiseWhite.value
      noiseSeed = lightning_Noise.spinnerNoiseSeed.value

      forkLocationMin = lightning_Forks.spinnerForksLocationMin.value
      forkLocationMax = lightning_Forks.spinnerForksLocationMax.value
			forkLengthMin = lightning_Forks.spinnerForksLengthMin.value
			forkLengthMax = lightning_Forks.spinnerForksLengthMax.value
			forkAngleMin = lightning_Forks.spinnerForksAngleMin.value
			forkAngleMax = lightning_Forks.spinnerForksAngleMax.value
			forkIterations = lightning_Forks.spinnerIterations.value
			forkBranchesMin = lightning_Forks.spinnerBranchesMin.value
			forkBranchesMax = lightning_Forks.spinnerBranchesMax.value
			forkBranchLengthMin = lightning_Forks.spinnerBranchLengthMin.value
			forkBranchLengthMax = lightning_Forks.spinnerBranchLengthMax.value
			
			if lightning_Forks.checkboxEnableForks.checked == false then
			  forkAmount = 0
			else
				forkAmount = lightning_Forks.spinnerForks.value

      animateNoise = lightning_Animation.checkboxAnimateNoise.checked
			animationSpeed = lightning_Animation.spinnerSpeed.value
			keyInterval =	lightning_Animation.spinnerKeys.value
			
			startTime = lightning_Animation.spinnerStartTime.value
			endTime = lightning_Animation.spinnerEndTime.value
			


			-- Local values
			local strikeDurMin = lightning_Animation.spinnerStrikeDurMin.value
			local strikeDurMax = lightning_Animation.spinnerStrikeDurMax.value
			
			-- Static, no animation
			if lightning_Animation.checkboxAnimateLightning.checked == false then (
				animateNoise = false

				for targetObject in targetObjs do (
			    createLightningStrike sourceObj.pos targetObject.pos progBar 0 0
				)

			-- Animate, strikes per object
			) else if lightning_Animation.checkboxStrikesPerObject.checked == true then (
				strikeArray = #()
				for targetObject in targetObjs do (
        	for time in startTime to endTime do (
           	-- Attempt to new lightning strikes when needed
						iterations = lightning_Animation.spinnerSimStrikes.value - strikeArray.count
        	 	for i in 1 to iterations do (
           		if (random 0.0 1.0) <= lightning_Animation.spinnerSrikeChance.value then (
	           		end = time + (random strikeDurMin strikeDurMax)
	           		if end > endTime then end = endTime
						    createLightningStrike sourceObj.pos targetObject.pos progBar time end
						    append strikeArray end
						  )
						)
						-- Remove lightning strikes from strikeArray when they are expired
						i = 1
						while i <= strikeArray.count do (
							if strikeArray[i] <= time then (
								deleteItem strikeArray i
							) else (
								i = i + 1
							)
						)
					)
				)

			-- Animate, strikes total
			) else (
				strikeArray = #()
       	for time in startTime to endTime do (
         	-- Attempt to new lightning strikes when needed
						iterations = lightning_Animation.spinnerSimStrikes.value - strikeArray.count
        	 	for i in 1 to iterations do (
	       		if (random 0.0 1.0) <= lightning_Animation.spinnerSrikeChance.value then (
           		end = time + (random strikeDurMin strikeDurMax)
           		if end > endTime then end = endTime
					    createLightningStrike sourceObj.pos targetObjs[(random 1 targetObjs.count)].pos progBar time end
					    append strikeArray end
					  )
					)
					-- Remove lightning strikes from strikeArray when they are expired
					i = 1
					while i <= strikeArray.count do (
						if strikeArray[i] <= time then (
							deleteItem strikeArray i
						) else (
							i = i + 1
						)
					)
				)
			)


 		) -- end lightning creation
 	)
)


global lightning_floater
try(closeRolloutFloater lightning_floater)catch()
lightning_floater = newrolloutfloater "Lightning" 161 530
addrollout lightning_Main lightning_floater
addrollout lightning_Spline lightning_floater
addrollout lightning_Forks lightning_floater
addrollout lightning_Noise lightning_floater
addrollout lightning_Animation lightning_floater